'use strict';

/* --------------------------------------------------------------
 swiper.js 2020-06-02
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/* globals Swiper */

/**
 * Widget that binds the swiper plugin (third party) to a DOM element
 *
 * @todo Remove the try - catch blocks and and correct the swiper issues.
 */
gambio.widgets.module('swiper', [gambio.source + '/libs/events', gambio.source + '/libs/responsive'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $body = $('body'),
        $slides = null,
        $controls = null,
        $target = null,
        $template = null,
        init = true,
        swiper = null,
        sliderOptions = null,
        hasThumbnails = true,
        mode = null,
        breakpointDataset = null,
        duplicates = false,
        preventSlideStart = false,
        sliderDefaults = { // Default configuration for the swiper
        pagination: '.swiper-pagination',
        nextButton: '.swiper-button-next',
        prevButton: '.swiper-button-prev',
        paginationClickable: true,
        loop: true,
        autoplay: 3,
        autoplayDisableOnInteraction: false
    },
        defaults = {
        // JSON that gets merged with the sliderDefaults and is passed to "swiper" directly.
        sliderOptions: null,
        // If this instance is a "main" swiper, the given selector selects the "control" swiper.
        controls: null,
        // If this instance is a "control" swiper, the given selector selects the "main" swiper.
        target: null,
        // Sets the initial slide (needed to prevent different init slides in main/controller slider).
        initSlide: null,
        // Detect if a swiper is needed for the breakpoint. If not, turn it off
        autoOff: false,
        // The translucence fix enables support for a fade effect between images with different aspect ratio,
        // but causing a delay between the change
        disableTranslucenceFix: false,
        breakpoints: [{
            // Until which breakpoint this settings is available
            breakpoint: 40,
            // If true, the paging bullets will be replaced with the preview images.
            usePreviewBullets: false,
            // This and all other settings belonging to the swiper plugin.
            slidesPerView: 2,
            // If true, the current slide gets centered in view (most usefull with an even slidesPerView
            // count).
            centeredSlides: true
        }, {
            breakpoint: 60,
            usePreviewBullets: true,
            slidesPerView: 3
        }, {
            breakpoint: 80,
            usePreviewBullets: true,
            slidesPerView: 3
        }, {
            breakpoint: 100,
            usePreviewBullets: true,
            slidesPerView: 5
        }]
    },
        options = $.extend({}, defaults, data),
        module = {};

    // ########## HELPER FUNCTIONS ##########

    /**
     * Function that generates the markup for
     * the preview bullets
     * @param       {Swiper}        swiper          Swiper object
     * @param       {integer}       index           Index of the slide
     * @param       {string}        className       The classname that must be add to the markup
     * @return      {string}                        The preview image html string
     * @private
     */
    var _generatePreviewButtons = function _generatePreviewButtons(swiper, index, className) {
        var $currentSlide = $slides.eq(index),
            $image = $currentSlide.find('img'),
            altTxt = $image.attr('alt'),
            thumbImage = $currentSlide.data('thumbImage');

        if (thumbImage) {
            return '<img src="' + thumbImage + '" alt="' + altTxt + '" class="' + className + '" />';
        }

        return '';
    };

    /**
     * Helper function to get the index of the
     * active slide
     * @return     {integer}                       The index of the active slide
     * @private
     */
    var _getIndex = function _getIndex() {
        var index = $this.find('.swiper-slide-active').index();

        // If there are duplicate slides (generated
        // by the swiper) recalculate the index
        index = duplicates ? index - 1 : index;
        index = index || 0;

        return index;
    };

    /**
     * Helper function to add the active
     * class to the active slide
     * @param       {integer}           index       The index of the active slide
     * @private
     */
    var _setActive = function _setActive(index) {
        $slides = $this.find('.swiper-slide:not(.swiper-slide-duplicate)');
        index = duplicates ? index + 1 : index;
        $slides.removeClass('active').eq(index).addClass('active');
    };

    // ########## EVENT HANDLER ##########

    /**
     * Event handler for the mouseenter event.
     * It disables the autoplay
     * @private
     */
    var _mouseEnterHandler = function _mouseEnterHandler() {
        try {
            if (swiper) {
                swiper.stopAutoplay();
            }
        } catch (e) {
            // Do not log the error
        }
    };

    /**
     * Event handler for the mouseleave event.
     * It enables the autoplay
     * @private
     */
    var _mouseLeaveHandler = function _mouseLeaveHandler() {
        try {
            if (swiper) {
                swiper.startAutoplay();
            }
        } catch (e) {
            // Do not log the error
        }
    };

    /**
     * Event handler for the goto event.
     * It switches the current slide to the given index
     * and adds the active class to the new active slide
     * @param       {object}    e       jQuery event object
     * @param       {number}    d       Index of the slide to show
     * @private
     */
    var _gotoHandler = function _gotoHandler(e, d) {
        e.stopPropagation();

        // Set the active slide
        _setActive(d);

        // Temporary deactivate the onSlideChangeStart event
        // to prevent looping through the goto / changeStart
        // events
        preventSlideStart = true;

        // Remove the autoplay after a goto event
        $this.off('mouseleave.swiper');
        swiper.stopAutoplay();

        // Try to correct the index between sliders
        // with and without duplicates
        var index = duplicates ? d + 1 : d;
        if (index > $slides.length) {
            index = 0;
        }

        // Goto the desired slide
        swiper.slideTo(index);

        // Reactivate the onSlideChangeEvent
        preventSlideStart = false;
    };

    /**
     * Click event handler that triggers a
     * "goto" event to the target swiper
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _clickHandler = function _clickHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var $self = $(this),
            index = $self.index();

        index = duplicates ? index - 1 : index;

        // Set the active slide
        _setActive(index);

        // Inform the main swiper
        $target.trigger(jse.libs.theme.events.SWIPER_GOTO(), index);
    };

    /**
     * Event that gets triggered on slideChange.
     * If the slide gets changed, the controls
     * will follow the current slide in position
     * @private
     */
    var _triggerSlideChange = function _triggerSlideChange() {
        if (!preventSlideStart) {
            var index = _getIndex(),
                lastIndex = $slides.closest(".swiper-slide-prev").index() || $slides.length - 2;

            var previousSlider = $slides.closest(".swiper-slide-prev");
            if (previousSlider) {
                lastIndex = duplicates ? previousSlider.index() - 1 : previousSlider.index();
            }

            if (index > $slides.length - 1) {
                index = 0;
            }

            // Recalculate index if duplicate slides are inside the slider
            if (index < 0) {
                index = $slides.length + index;
            } else {
                index = duplicates && index === lastIndex ? index - lastIndex : index;
            }

            // Set the active slide
            _setActive(index);

            // Inform the controls
            $controls.trigger(jse.libs.theme.events.SWIPER_GOTO(), index);
        }
    };

    /**
     * Workaround for the translucence issue
     * that happens on small screens with enabled
     * fade effect. Maybe it can be removed, if the
     * swiper gets updated itself
     * @private
     */
    var _translucenceWorkaround = function _translucenceWorkaround() {
        if (!options.disableTranslucenceFix && sliderOptions && sliderOptions.effect === 'fade') {
            $this.find('.swiper-slide').filter(':not(.swiper-slide-active)').fadeTo(300, 0, function () {
                $(this).css('visibility', 'hidden');
            });

            $this.find('.swiper-slide').filter('.swiper-slide-active').fadeTo(300, 1, function () {
                $(this).css('visibility', '');
            });
        }
    };

    /**
     * The breakpoint handler initializes the swiper
     * with the settings for the current breakpoint.
     * Therefore it uses the default slider options,
     * the custom slider options given by the options
     * object and the breakpoint options object also
     * given by the options (in this order)
     * @private
     */
    var _breakpointHandler = function _breakpointHandler() {

        // Get the current viewmode
        var oldMode = mode || {},
            newMode = jse.libs.theme.responsive.breakpoint(),
            extendOptions = options.breakpoints[0] || {},
            newBreakpointDataset = null;

        // Only do something if the view was changed
        if (newMode.id !== oldMode.id) {

            // Store the new viewmode
            mode = $.extend({}, newMode);

            // Iterate through the breakpoints object to detect
            // the correct settings for the current breakpoint
            $.each(options.breakpoints, function (i, v) {
                if (v.breakpoint > newMode.id) {
                    return false;
                }
                newBreakpointDataset = i;
                extendOptions = v;
            });

            if (options.sliderOptions && options.sliderOptions.breakpoints) {
                $.each(options.sliderOptions.breakpoints, function (i, v) {
                    if (v.breakpoint === newMode.id) {
                        extendOptions = v;
                        return false;
                    }
                });
            }

            // Only do something if the settings change due browser
            // resize or if it's the first time run
            if (newBreakpointDataset !== breakpointDataset || init) {
                // Combine the settings
                sliderOptions = $.extend({}, sliderDefaults, options.sliderOptions || {}, extendOptions);

                // Add the preview image bullets function to the options object
                if (sliderOptions.usePreviewBullets && hasThumbnails) {
                    sliderOptions.paginationBulletRender = _generatePreviewButtons;
                }

                // Add the autoplay interval to the options object
                sliderOptions.autoplay = sliderOptions.autoplay ? sliderOptions.autoplay * 1000 : 0;

                // Disable loop if there is only one slider.
                if ($this.find('.swiper-slide').length === 1) {
                    sliderOptions.loop = false;
                }

                // If an swiper exists, get the current
                // slide no. and remove the old swiper
                if (swiper) {
                    sliderOptions.initialSlide = _getIndex();
                    try {
                        swiper.destroy(true, true);
                    } catch (ignore) {
                        swiper = null;
                    }
                } else {
                    sliderOptions.initialSlide = options.initSlide || sliderOptions.initialSlide || 0;
                }

                var $duplicate = $this.find('.swiper-slide:not(.swiper-slide-duplicate)');

                if (!options.autoOff || $duplicate.length > sliderOptions.slidesPerView && options.autoOff) {
                    $this.addClass('swiper-is-active').removeClass('swiper-is-not-active');

                    // Initialize the swiper
                    try {
                        swiper = new Swiper($this, sliderOptions);
                    } catch (e) {
                        return; // Swiper might throw an error upon initialization that should not halt the script execution.
                    }

                    swiper.off('onTransitionEnd onSlideChangeStart').on('onTransitionEnd', _translucenceWorkaround);

                    // If this is a "main" swiper and has external controls, an
                    // goto event is triggered if the current slide is changed
                    if ($controls.length) {
                        swiper.on('onSlideChangeStart', _triggerSlideChange);
                    }

                    // Add the event handler
                    $this.off('mouseenter.swiper mouseleave.swiper ' + jse.libs.theme.events.SWIPER_GOTO() + ' ' + jse.libs.theme.events.SLIDES_UPDATE()).on('mouseenter.swiper', _mouseEnterHandler).on('mouseleave.swiper', _mouseLeaveHandler).on(jse.libs.theme.events.SWIPER_GOTO(), _gotoHandler).on(jse.libs.theme.events.SLIDES_UPDATE(), _updateSlides);

                    if (init) {
                        // Check if there are duplicates slides (generated by the swiper)
                        // after the first time init of the swiper
                        duplicates = !!$this.find('.swiper-slide-duplicate').length;
                    }

                    // Set the active slide
                    var index = init && options.initSlide ? options.initSlide : _getIndex();
                    _setActive(index);

                    // Inform the controls that the main swiper has changed
                    // In case that the other slider isn't initialized yet,
                    // set an data attribute to the markup element to inform
                    // it on init
                    if ($controls.length) {
                        $controls.attr('data-swiper-init-slide', index);
                        _triggerSlideChange();
                    }

                    // Unset the init flag
                    init = false;

                    setTimeout(function () {
                        return swiper.update();
                    });
                } else {
                    // Disable the swiper buttons
                    $this.removeClass('swiper-is-active').addClass('swiper-is-not-active');
                    init = true;
                }
            }
        }
    };

    /**
     * Event handler that adds & removes slides from the
     * swiper. After the slides were processed, the first
     * slide is shown
     * @param       {object}    e       jQuery event object
     * @param       {object}    d       JSON object that contains the categories / images
     * @private
     */
    var _updateSlides = function _updateSlides(e, d) {

        // Loops through each category inside the images array
        $.each(d, function (category, dataset) {
            var catName = category + '-category',
                add = [],
                remove = [],
                markup = $template.html();

            // Get all indexes from the slides
            // of the same category and remove
            // them from the slider
            $slides.filter('.' + catName).each(function () {
                var $self = $(this),
                    index = $self.data().swiperSlideIndex;

                index = index === undefined ? $self.index() : index;
                remove.push(index);
            });
            swiper.removeSlide(remove);

            // Generate the markup for the new slides
            // and add them to the slider
            $.each(dataset || [], function (i, v) {
                v.className = catName;
                v.srcattr = 'src="' + v.src + '"';
                add.push(Mustache.render(markup, v));
            });
            swiper.appendSlide(add);
        });

        $slides = $this.find('.swiper-slide');

        // To prevent an inconsistent state
        // in control / main slider combinations
        // slide to the first slide
        _setActive(0);
        var index = duplicates ? 1 : 0;
        swiper.slideTo(index, 0);
    };

    /**
     * Prevent text selection by clicking on swiper buttons
     * @private
     */
    var _preventTextSelection = function _preventTextSelection() {
        $(options.sliderOptions.nextButton).on('selectstart', function () {
            return false;
        });
        $(options.sliderOptions.prevButton).on('selectstart', function () {
            return false;
        });
    };

    /**
     * Sets the initial height for one swiper image container to prevent cut off images on smaller swiper heights
     * @private
     */
    var _scaleThumbnailHeight = function _scaleThumbnailHeight() {
        var swiperContainer = $('.swiper-container-vertical .swiper-slide');
        var $containerHeight = swiperContainer.css('height');

        // Workaround for IE Browsers
        if ($('.swiper-container-vertical').hasClass('swiper-wp8-vertical')) {
            $containerHeight = swiperContainer.height() + 5;

            swiperContainer.css('height', $containerHeight);
        }

        if ($containerHeight === '0px') {
            $containerHeight = $('.product-info-thumbnails-mobile').css('height');
        }

        $('.align-middle').css('height', $containerHeight);
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        $slides = $this.find('.swiper-slide');
        $controls = $(options.controls);
        $target = $(options.target);
        $template = $this.find('template');

        // Check if all images inside the swiper have
        // thumbnail image given
        $slides.each(function () {
            if (!$(this).data().thumbImage) {
                hasThumbnails = false;
                return false;
            }
        });

        // Add the breakpoint handler ty dynamically
        // set the options corresponding to the browser size (slider responsive will re-initialize the swiper).
        _breakpointHandler();

        // If this instance is a "control" swiper the target is the main swiper
        // which will be updated on a click inside this control swiper
        if (options.target) {
            $this.on('click.swiper', '.swiper-slide', _clickHandler);
        }

        $(document).ready(function () {
            $('.swiper-vertical .swiper-slide[data-index]').css('display', 'inline-block');
            $('.product-info-image .swiper-slide[data-index]').css('z-index', 'inherit');
            $('.product-info-image .swiper-slide[data-index] .swiper-slide-inside img.img-responsive').fadeIn(1000);
        });

        _translucenceWorkaround();
        _preventTextSelection();
        _scaleThumbnailHeight();

        // Fix for invisible Thumbnail-Images for switching from Tablet-Portrait to Tablet-Landscape
        $body.on(jse.libs.theme.events.BREAKPOINT(), function () {
            _scaleThumbnailHeight();
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
